using System;
using System.Collections.Generic;

class InversionCountAVLTree
{
    public class Node
    {
        public float key;
        public Node left, right;
        public int size, height;

        public Node(float val)
        {
            key = val;
            left = null;
            right = null;
            height = 1;
            size = 1;
        }
    }

    public int Result { get; private set; }
    private Node root;
    private Stack<Node> helperStack;

    public InversionCountAVLTree()
    {
        root = null;
        helperStack = new Stack<Node>();
    }

    static int Height(Node N)
    {
        if (N == null)
            return 0;
        return N.height;
    }

    static int Size(Node N)
    {
        if (N == null)
            return 0;
        return N.size;
    }

    static Node RightRotate(Node y)
    {
        var x = y.left;
        var T2 = x.right;

        x.right = y;
        y.left = T2;

        y.height = Math.Max(Height(y.left), Height(y.right)) + 1;
        x.height = Math.Max(Height(x.left), Height(x.right)) + 1;

        y.size = Size(y.left) + Size(y.right) + 1;
        x.size = Size(x.left) + Size(x.right) + 1;

        return x;
    }

    static Node LeftRotate(Node x)
    {
        var y = x.right;
        var T2 = y.left;

        y.left = x;
        x.right = T2;

        x.height = Math.Max(Height(x.left), Height(x.right)) + 1;
        y.height = Math.Max(Height(y.left), Height(y.right)) + 1;

        x.size = Size(x.left) + Size(x.right) + 1;
        y.size = Size(y.left) + Size(y.right) + 1;

        return y;
    }

    static int GetBalance(Node N)
    {
        if (N == null)
            return 0;
        return Height(N.left) - Height(N.right);
    }

    public void Insert(float key)
    {
        Node current = root;
        while (current != null)
        {
            helperStack.Push(current);

            if (key <= current.key)
            {
                Result += Size(current.right) + 1;
                current = current.left;
            }
            else
            {
                current = current.right;
            }
        }

        if (helperStack.Count == 0)
        {
            root = new Node(key);
            return;
        }

        Node newNode = new Node(key);
        Node parent = helperStack.Pop();

        if (key <= parent.key)
        {
            parent.left = newNode;
        }
        else
        {
            parent.right = newNode;
        }

        while (helperStack.Count > 0)
        {
            Node ancestor = helperStack.Pop();
            ancestor.height = Math.Max(Height(ancestor.left), Height(ancestor.right)) + 1;
            ancestor.size = Size(ancestor.left) + Size(ancestor.right) + 1;

            int balance = GetBalance(ancestor);

            if (balance > 1 && key < ancestor.left.key)
            {
                ancestor = RightRotate(ancestor);
            }
            else if (balance < -1 && key > ancestor.right.key)
            {
                ancestor = LeftRotate(ancestor);
            }
            else if (balance > 1 && key > ancestor.left.key)
            {
                ancestor.left = LeftRotate(ancestor.left);
                ancestor = RightRotate(ancestor);
            }
            else if (balance < -1 && key < ancestor.right.key)
            {
                ancestor.right = RightRotate(ancestor.right);
                ancestor = LeftRotate(ancestor);
            }

            if (helperStack.Count == 0)
            {
                root = ancestor;
            }
            else
            {
                Node grandparent = helperStack.Peek();

                if (ancestor.key <= grandparent.key)
                {
                    grandparent.left = ancestor;
                }
                else
                {
                    grandparent.right = ancestor;
                }
            }
        }
    }


    public void Clear()
    {
        if (root == null)
        {
            return;
        }

        helperStack.Push(root);

        while (helperStack.Count > 0)
        {
            var current = helperStack.Pop();

            if (current.right != null)
            {
                helperStack.Push(current.right);
            }

            if (current.left != null)
            {
                helperStack.Push(current.left);
            }

            current.left = null;
            current.right = null;
        }
        
        root = null;
        Result = 0;
    }
}

class Program
{
    class Trkac
    {
        public int Brzina { get; private set; }
        public int Pozicija { get; private set; }

        public Trkac(int pozicija, int brzina)
        {
            Brzina = brzina;
            Pozicija = pozicija;
        }

        public float VremeDoPozicije(float pozicija)
        {
            return (pozicija - Pozicija) / Brzina;
        }

        public float NovaPozicija(float vreme)
        {
            return Pozicija + vreme * Brzina;
        }
    }
    
    const float preciznost = 0.000001f;
    static InversionCountAVLTree drvo = new InversionCountAVLTree();

    static int BrojPreticanjaNaPoziciji(int brojTrkaca, Trkac[] trkaci, float pozicija)
    {
        var vreme = trkaci[0].VremeDoPozicije(pozicija);

        for(int i = 0; i < brojTrkaca ; i++)
        {
            var zaDodavanje = trkaci[i].NovaPozicija(vreme);
            drvo.Insert(zaDodavanje);
        }
        var preticanja = drvo.Result;
        drvo.Clear();
        return preticanja;
    }

    static float VremeDoPreticanja(int brojTrkaca, Trkac[] trkaci, int brojPreticanja)
    {
        var levi = 0f;
        var desni = float.MaxValue;
        var srednji = 0f;
        var nulti = trkaci[0];

        while ((desni - levi) > preciznost)
        {
            srednji = (desni - levi) / 2 + levi;
            var preticanja = BrojPreticanjaNaPoziciji(brojTrkaca, trkaci, srednji);

            if (preticanja < brojPreticanja)
            {
                levi = srednji + preciznost;
            }
            else
            {
                desni = srednji - preciznost;
            }
        }

        return nulti.VremeDoPozicije(srednji);
    }

    static void Main()
    {
        var dimenzija = Console.ReadLine().Split(' ');
        var brojTrkaca = int.Parse(dimenzija[0]);
        var brojPreticanja = int.Parse(dimenzija[1]);

        var trkaci = new Trkac[brojTrkaca];
        for(int i = 0; i < brojTrkaca; i++)
        {
            var podaci = Console.ReadLine().Split(' ');
            trkaci[i] = new Trkac(int.Parse(podaci[0]), int.Parse(podaci[1]));
        }

        var resenje = VremeDoPreticanja(brojTrkaca, trkaci, brojPreticanja);
        Console.WriteLine(resenje.ToString(".000000000"));
    }
}